import React, { useState, useEffect } from 'react';
import { getImageUrl } from '../../../utils/image-helper';
import { formatCurrency } from '../../../utils/currency-formatter';
import { CheckoutProvider, useCheckoutContext } from '../../../contexts/CheckoutContext';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

interface Product {
  id: string;
  name: string;
  price: number;
  image: string;
  sku: string;
  taxName?: string;
  taxPercentage?: number;
}

interface CartItem extends Product {
  quantity: number;
  variants?: {[key: string]: string} | null;
}

interface CheckoutProps {
  cartItems: CartItem[];
  currency: string;
  onClose: () => void;
  onOrderComplete: () => void;
  onUpdateCart: (items: CartItem[]) => void;
  userProfile?: {
    first_name: string;
    last_name: string;
    email: string;
    phone: string;
    address: string;
    city: string;
    state: string;
    country: string;
    postalCode: string;
  };
  isLoggedIn?: boolean;
  onRemoveFromCart: (index: number) => void;
  onUpdateQuantity: (index: number, change: number) => void;
  onQuantityChange: (index: number, value: string) => void;
  store?: { id: string; slug: string };
}

const CheckoutContent: React.FC<Omit<CheckoutProps, 'userProfile' | 'isLoggedIn' | 'store'>> = ({ cartItems, currency, onClose, onOrderComplete, onUpdateCart, onRemoveFromCart, onUpdateQuantity, onQuantityChange }) => {
  const {
    step,
    customerInfo,
    couponCode,
    appliedCoupon,
    couponError,
    selectedShipping,
    selectedPayment,
    shippingMethods,
    loadingShipping,
    paymentMethods,
    loadingPayments,
    emailError,
    phoneError,
    isPlacingOrder,
    countryId,
    stateId,
    cityId,
    CountryDropdown,
    StateDropdown,
    CityDropdown,
    setCouponCode,
    setSelectedShipping,
    setSelectedPayment,
    handleInputChange,
    handleNextStep,
    handlePrevStep,
    handleApplyCoupon,
    handlePlaceOrder,
    setCountryId,
    setStateId,
    setCityId,
    setBankTransferFile,
    whatsappNumber,
    whatsappError,
    setWhatsappNumber,
    setWhatsappError
  } = useCheckoutContext();

  // Get store settings for currency formatting
  const storeSettings = (window as any).page?.props?.storeSettings || {};
  const currencies = (window as any).page?.props?.currencies || [];
  
  // Calculate totals
  const subtotal = cartItems.reduce((sum, item) => sum + (item.price * item.quantity), 0);
  const totalTax = cartItems.reduce((sum, item) => {
    const itemTotal = item.price * item.quantity;
    const taxAmount = item.taxPercentage ? (itemTotal * item.taxPercentage) / 100 : 0;
    return sum + taxAmount;
  }, 0);
  const couponDiscount = appliedCoupon ? Number(appliedCoupon.discount) || 0 : 0;
  const selectedShippingMethod = shippingMethods.find(method => method.id.toString() === selectedShipping);
  const shippingCost = selectedShippingMethod ? 
    (selectedShippingMethod.type === 'percentage_based' ? 
      (subtotal * parseFloat(selectedShippingMethod.cost || 0)) / 100 : 
    selectedShippingMethod.type === 'free' ? 0 :
      parseFloat(selectedShippingMethod.cost || 0)) : 0;
  const total = subtotal + totalTax - couponDiscount + shippingCost;

  return (
    <div className="fixed inset-0 z-50 overflow-hidden">
      <div className="absolute inset-0 bg-black/40"></div>
      <div className="absolute inset-0 flex items-center justify-center p-0 md:p-4">
        <div className="bg-white rounded-2xl shadow-2xl w-full max-w-4xl h-[90vh] flex flex-col overflow-hidden" onClick={(e) => e.stopPropagation()}>
          {/* Header */}
          <div className="flex items-center justify-between p-4 md:p-6 border-b border-gray-100 flex-shrink-0">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-blue-100 rounded-lg">
                <svg className="w-5 h-5 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 11V7a4 4 0 00-8 0v4M5 9h14l-1 12H6L5 9z" />
                </svg>
              </div>
              <h2 className="text-xl font-bold text-gray-900">Checkout</h2>
            </div>
            <button onClick={onClose} className="p-2 hover:bg-gray-100 rounded-lg transition-colors cursor-pointer">
              <svg className="w-5 h-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
              </svg>
            </button>
          </div>

          {/* Progress Steps */}
          <div className="px-4 md:px-6 py-4 bg-gray-50 border-b border-gray-100 flex-shrink-0">
            <div className="flex items-center justify-center space-x-4 md:space-x-8">
              {[1, 2, 3].map((stepNum) => (
                <div key={stepNum} className="flex items-center">
                  {/* Mobile: Vertical Layout */}
                  <div className="flex flex-col items-center md:hidden">
                    <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-semibold ${
                      step >= stepNum ? 'bg-blue-600 text-white' : 'bg-gray-200 text-gray-600'
                    }`}>
                      {stepNum}
                    </div>
                    <span className={`mt-1 text-xs ${step >= stepNum ? 'text-blue-600 font-semibold' : 'text-gray-500'}`}>
                      {stepNum === 1 ? 'Info' : stepNum === 2 ? 'Review' : 'Payment'}
                    </span>
                  </div>
                  
                  {/* Desktop: Horizontal Layout */}
                  <div className="hidden md:flex md:items-center">
                    <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-semibold ${
                      step >= stepNum ? 'bg-blue-600 text-white' : 'bg-gray-200 text-gray-600'
                    }`}>
                      {stepNum}
                    </div>
                    <span className={`ml-2 text-sm ${step >= stepNum ? 'text-blue-600 font-semibold' : 'text-gray-500'}`}>
                      {stepNum === 1 ? 'Information' : stepNum === 2 ? 'Review' : 'Payment'}
                    </span>
                  </div>
                  
                  {stepNum < 3 && <div className={`w-6 md:w-16 h-0.5 ml-2 md:ml-4 ${step > stepNum ? 'bg-blue-600' : 'bg-gray-200'}`}></div>}
                </div>
              ))}
            </div>
          </div>

          {/* Content */}
          <div className="flex-1 overflow-y-auto">
            {step === 1 && (
              <div className="p-4 md:p-6">
                <h3 className="text-lg font-semibold text-gray-900 mb-4 md:mb-6">Customer/Shipping Information</h3>
                <div className="grid grid-cols-1 gap-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">First Name *</label>
                      <input
                        type="text"
                        value={customerInfo.firstName}
                        onChange={(e) => handleInputChange('firstName', e.target.value)}
                        className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-base"
                        placeholder="Enter first name"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">Last Name *</label>
                      <input
                        type="text"
                        value={customerInfo.lastName}
                        onChange={(e) => handleInputChange('lastName', e.target.value)}
                        className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-base"
                        placeholder="Enter last name"
                      />
                    </div>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Email *</label>
                    <input
                      type="email"
                      value={customerInfo.email}
                      onChange={(e) => handleInputChange('email', e.target.value)}
                      className={`w-full px-4 py-3 border rounded-lg focus:outline-none focus:ring-2 text-base ${
                        emailError ? 'border-red-500 focus:ring-red-500' : 'border-gray-300 focus:ring-blue-500'
                      }`}
                      placeholder="Enter your email"
                    />
                    {emailError && <p className="text-red-500 text-sm mt-1">{emailError}</p>}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Phone *</label>
                    <input
                      type="tel"
                      value={customerInfo.phone}
                      onChange={(e) => handleInputChange('phone', e.target.value)}
                      className={`w-full px-4 py-3 border rounded-lg focus:outline-none focus:ring-2 text-base ${
                        phoneError ? 'border-red-500 focus:ring-red-500' : 'border-gray-300 focus:ring-blue-500'
                      }`}
                      placeholder="Enter your phone number"
                    />
                    {phoneError && <p className="text-red-500 text-sm mt-1">{phoneError}</p>}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Address *</label>
                    <textarea
                      value={customerInfo.address}
                      onChange={(e) => handleInputChange('address', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-base"
                      rows={3}
                      placeholder="Enter your full address"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Country *</label>
                    <Select
                      value={(() => {
                        const countries = (window as any).page?.props?.countries || [];
                        const selectedCountry = countries.find(c => c.name === customerInfo.country);
                        return selectedCountry?.id?.toString() || '';
                      })()} 
                      onValueChange={(countryId) => {
                        const countries = (window as any).page?.props?.countries || [];
                        const selectedCountry = countries.find(c => c.id.toString() === countryId);
                        if (selectedCountry) {
                          handleInputChange('country', selectedCountry.name);
                          handleInputChange('state', '');
                          handleInputChange('city', '');
                          setCountryId(selectedCountry.id);
                          setStateId(undefined);
                          setCityId(undefined);
                        }
                      }}
                    >
                      <SelectTrigger className="w-full focus:ring-blue-500 focus:border-blue-500">
                        <SelectValue placeholder="Select Country" />
                      </SelectTrigger>
                      <SelectContent>
                        {((window as any).page?.props?.countries || []).map(country => (
                          <SelectItem key={country.id} value={country.id.toString()}>
                            {country.name}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">State *</label>
                    <StateDropdown
                      countryId={countryId}
                      value={customerInfo.state}
                      onChange={(value, id) => {
                        handleInputChange('state', value);
                        handleInputChange('city', '');
                        setStateId(id);
                        setCityId(undefined);
                      }}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-base"
                      disabled={!countryId}
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">City *</label>
                    <CityDropdown
                      stateId={stateId}
                      value={customerInfo.city}
                      onChange={(value, id) => {
                        handleInputChange('city', value);
                        setCityId(id);
                      }}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-base"
                      disabled={!stateId}
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Postal Code *</label>
                    <input
                      type="text"
                      value={customerInfo.postalCode}
                      onChange={(e) => handleInputChange('postalCode', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-base"
                      placeholder="Enter postal code"
                    />
                  </div>
                </div>
              </div>
            )}

            {step === 2 && (
              <div className="p-4 md:p-6">
                <h3 className="text-lg font-semibold text-gray-900 mb-4 md:mb-6">Order Review</h3>
                <div className="space-y-4 mb-6">
                  {cartItems.map((item, index) => (
                    <div key={index} className="bg-white rounded-2xl p-4 shadow-sm border border-gray-100">
                      <div className="flex items-start gap-4">
                        <img src={getImageUrl(item.image)} alt={item.name} loading="lazy" className="w-16 h-16 object-cover rounded-xl" />
                        <div className="flex-1 min-w-0">
                          <h4 className="font-semibold text-gray-900 mb-1 text-sm leading-tight">{item.name}</h4>
                          {(() => {
                            const variants = typeof item.variants === 'string' ? JSON.parse(item.variants) : item.variants;
                            return variants && Object.keys(variants).length > 0 && (
                              <div className="text-xs text-gray-500 mb-1">
                                {Object.entries(variants).map(([key, value], index) => (
                                  <span key={key}>
                                    {key}: {value}
                                    {index < Object.keys(variants).length - 1 && ', '}
                                  </span>
                                ))}
                              </div>
                            );
                          })()}
                          <div className="flex items-center justify-between mb-2">
                            <p className="text-blue-600 font-bold text-lg">{formatCurrency(item.price * item.quantity, storeSettings, currencies)}</p>
                            <button 
                              onClick={() => onRemoveFromCart(index)}
                              className="p-1.5 text-red-500 hover:text-red-700 hover:bg-red-50 rounded-lg transition-colors cursor-pointer"
                            >
                              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1-1H8a1 1 0 00-1 1v3M4 7h16" />
                              </svg>
                            </button>
                          </div>
                          <div className="flex items-center justify-between">
                            <span className="text-xs text-gray-500">{formatCurrency(item.price, storeSettings, currencies)} each</span>
                            <div className="flex items-center gap-3 bg-gray-100 rounded-lg p-1">
                              <button 
                                onClick={() => item.quantity > 1 && onUpdateQuantity(index, -1)}
                                className={`w-7 h-7 flex items-center justify-center rounded-md text-white text-sm shadow-sm font-bold leading-none ${
                                  item.quantity > 1 ? 'bg-blue-600 hover:bg-blue-700 cursor-pointer' : 'bg-gray-400 cursor-not-allowed'
                                }`}
                              >
                                −
                              </button>
                              <input
                                type="number"
                                value={item.quantity}
                                onChange={(e) => onQuantityChange(index, e.target.value)}
                                className="w-8 text-sm font-semibold text-gray-900 text-center bg-transparent border-0 focus:outline-none [appearance:textfield] [&::-webkit-outer-spin-button]:appearance-none [&::-webkit-inner-spin-button]:appearance-none"
                                min="1"
                              />
                              <button 
                                onClick={() => onUpdateQuantity(index, 1)}
                                className="w-7 h-7 flex items-center justify-center bg-blue-600 hover:bg-blue-700 rounded-md text-white text-sm cursor-pointer shadow-sm font-bold leading-none"
                              >
                                +
                              </button>
                            </div>
                          </div>
                          {item.taxName && item.taxPercentage && (
                            <p className="text-xs text-gray-400 mt-2">
                              {item.taxName}: {item.taxPercentage}% ({formatCurrency((item.price * item.quantity * item.taxPercentage) / 100, storeSettings, currencies)})
                            </p>
                          )}
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
                
                {/* Coupon Section */}
                <div className="bg-gray-50 rounded-lg p-4 mb-4">
                  <h4 className="font-semibold text-gray-900 mb-3">Apply Coupon</h4>
                  <div className="flex flex-col sm:flex-row gap-2">
                    <input
                      type="text"
                      value={couponCode}
                      onChange={(e) => {
                        setCouponCode(e.target.value);
                        setCouponError('');
                      }}
                      placeholder="Enter coupon code"
                      className={`flex-1 px-3 py-3 border rounded-lg focus:outline-none focus:ring-2 text-base bg-white ${
                        couponError ? 'border-red-500 focus:ring-red-500' : 'border-gray-300 focus:ring-blue-500'
                      }`}
                    />
                    <button
                      onClick={() => handleApplyCoupon(subtotal)}
                      className="px-4 py-3 bg-blue-600 hover:bg-blue-700 text-white rounded-lg transition-colors cursor-pointer font-medium text-sm sm:whitespace-nowrap"
                    >
                      Apply
                    </button>
                  </div>
                  {couponError && (
                    <div className="mt-2 text-sm text-red-600">
                      {couponError}
                    </div>
                  )}
                  {appliedCoupon && (
                    <div className="mt-2 text-sm text-green-600">
                      ✓ Coupon "{appliedCoupon.code}" applied
                    </div>
                  )}
                </div>

                {/* Shipping Section */}
                <div className="bg-gray-50 rounded-lg p-4 mb-4">
                  <h4 className="font-semibold text-gray-900 mb-3">Shipping Methods</h4>
                  
                  {loadingShipping ? (
                    <div className="text-center py-4">
                      <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600 mx-auto"></div>
                      <p className="text-sm text-gray-500 mt-2">Loading shipping options...</p>
                    </div>
                  ) : shippingMethods.length > 0 ? (
                    <div className="space-y-2">
                      {shippingMethods.map((method) => (
                        <label key={method.id} className="flex items-center gap-3 p-2 border rounded-lg cursor-pointer bg-white hover:bg-gray-50">
                          <input
                            type="radio"
                            name="shipping"
                            value={method.id}
                            checked={selectedShipping === method.id.toString()}
                            onChange={(e) => setSelectedShipping(e.target.value)}
                            className="text-blue-600"
                          />
                          <div className="flex-1">
                            <div className="font-medium text-sm">{method.name}</div>
                            <div className="text-xs text-gray-500">{method.delivery_time || 'Standard delivery'}</div>
                            {method.description && (
                              <div className="text-xs text-gray-400">{method.description}</div>
                            )}
                          </div>
                          <div className="font-semibold text-sm">
                            {method.type === 'percentage_based' ? 
                              `${parseFloat(method.cost || 0)}%` : 
                            method.type === 'free' ? 'Free' :
                              formatCurrency(parseFloat(method.cost || 0), storeSettings, currencies)
                            }
                          </div>
                        </label>
                      ))}
                    </div>
                  ) : (
                    <div className="text-center py-4">
                      <p className="text-sm text-gray-500">No shipping methods available</p>
                    </div>
                  )}
                </div>

                {/* Price Summary */}
                <div className="bg-gray-50 rounded-lg p-4">
                  <div className="space-y-2">
                    <div className="flex justify-between">
                      <span className="text-gray-600">Subtotal</span>
                      <span className="font-semibold">{formatCurrency(subtotal, storeSettings, currencies)}</span>
                    </div>
                    {totalTax > 0 && (
                      <div className="flex justify-between">
                        <span className="text-gray-600">Total Tax</span>
                        <span className="font-semibold">{formatCurrency(totalTax, storeSettings, currencies)}</span>
                      </div>
                    )}
                    {appliedCoupon && (
                      <div className="flex justify-between text-green-600">
                        <span>Coupon Discount{appliedCoupon.type === 'percentage' ? ` (${appliedCoupon.discount_amount}%)` : ''}</span>
                        <span>-{formatCurrency(couponDiscount, storeSettings, currencies)}</span>
                      </div>
                    )}
                    {shippingCost > 0 && (
                      <div className="flex justify-between">
                        <span className="text-gray-600">Shipping</span>
                        <span className="font-semibold">{formatCurrency(shippingCost, storeSettings, currencies)}</span>
                      </div>
                    )}
                    <div className="flex justify-between pt-2 border-t border-gray-200">
                      <span className="text-lg font-bold">Total</span>
                      <span className="text-xl font-bold text-blue-600">{formatCurrency(total, storeSettings, currencies)}</span>
                    </div>
                  </div>
                </div>
              </div>
            )}

            {step === 3 && (
              <div className="p-4 md:p-6">
                <h3 className="text-lg font-semibold text-gray-900 mb-4 md:mb-6">Payment Method</h3>
                
                {loadingPayments ? (
                  <div className="text-center py-8">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
                    <p className="text-sm text-gray-500 mt-2">Loading payment methods...</p>
                  </div>
                ) : paymentMethods.length > 0 ? (
                  <div className="space-y-3">
                    {paymentMethods.map((method) => (
                      <div key={method.name} className={`border-2 rounded-lg transition-all ${
                        selectedPayment === method.name ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:border-gray-300'
                      }`}>
                        <label className="flex items-center gap-4 p-4 cursor-pointer">
                          <input
                            type="radio"
                            name="payment"
                            value={method.name}
                            checked={selectedPayment === method.name}
                            onChange={(e) => setSelectedPayment(e.target.value)}
                            className="text-blue-600"
                          />
                          <div className="w-6 h-6 text-blue-600">
                            {method.icon ? (
                              <div dangerouslySetInnerHTML={{ __html: method.icon }} />
                            ) : (
                              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-3a2 2 0 00-2-2H9a2 2 0 00-2 2v3a2 2 0 002 2zm8-12V5a2 2 0 00-2-2H9a2 2 0 00-2 2v4h10z" />
                              </svg>
                            )}
                          </div>
                          <div className="flex-1">
                            <h4 className="font-semibold text-gray-900">{method.display_name}</h4>
                            {method.description && (
                              <p className="text-sm text-gray-600">{method.description}</p>
                            )}
                          </div>
                        </label>
                        
                        {selectedPayment === method.name && method.name === 'bank' && method.details && (
                          <div className="px-4 pb-4 border-t border-gray-200 mt-2 pt-4">
                            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-4">
                              <h5 className="font-semibold text-blue-900 mb-2">Bank Transfer Details</h5>
                              <div className="text-sm text-blue-800 whitespace-pre-line">{method.details}</div>
                            </div>
                            <div>
                              <label className="block text-sm font-semibold text-gray-800 mb-2">
                                Upload Payment Receipt *
                              </label>
                              <input
                                type="file"
                                accept="image/*,.pdf"
                                onChange={(e) => {
                                  const file = e.target.files?.[0];
                                  if (file) setBankTransferFile(file);
                                }}
                                className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 bg-white"
                              />
                            </div>
                          </div>
                        )}
                        
                        {selectedPayment === method.name && method.name === 'whatsapp' && (
                          <div className="px-4 pb-4 border-t border-gray-200 mt-2 pt-4">
                            <div className="bg-green-50 border border-green-200 rounded-lg p-4 mb-4">
                              <h5 className="font-semibold text-green-900 mb-2">WhatsApp Payment</h5>
                              <div className="text-sm text-green-800">You will be redirected to WhatsApp to complete your payment.</div>
                            </div>
                            <div>
                              <label className="block text-sm font-semibold text-gray-800 mb-2">
                                WhatsApp Number *
                              </label>
                              <input
                                type="tel"
                                value={whatsappNumber}
                                onChange={(e) => {
                                  setWhatsappNumber(e.target.value);
                                  setWhatsappError('');
                                }}
                                placeholder="+1234567890"
                                className={`w-full px-4 py-3 border-2 rounded-lg focus:outline-none bg-white ${
                                  whatsappError ? 'border-red-500 focus:border-red-500' : 'border-gray-300 focus:border-green-500'
                                }`}
                              />
                              {whatsappError && (
                                <p className="mt-1 text-sm text-red-600">{whatsappError}</p>
                              )}
                              <p className="mt-1 text-sm text-gray-600">
                                Enter your WhatsApp number with country code (e.g., +91 9876543210)
                              </p>
                            </div>
                          </div>
                        )}
                        
                        {selectedPayment === method.name && method.name !== 'bank' && method.name !== 'whatsapp' && method.form_fields && (
                          <div className="px-4 pb-4 border-t border-gray-200 mt-2 pt-4">
                            <div className="space-y-3">
                              {method.form_fields.map((field: any, index: number) => (
                                <div key={index}>
                                  <label className="block text-sm font-medium text-gray-700 mb-2">
                                    {field.label} {field.required && '*'}
                                  </label>
                                  {field.type === 'file' ? (
                                    <input
                                      type="file"
                                      accept={field.accept || "image/*,.pdf"}
                                      onChange={(e) => setBankTransferFile(e.target.files?.[0] || null)}
                                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100"
                                      required={field.required}
                                    />
                                  ) : (
                                    <input
                                      type={field.type || 'text'}
                                      placeholder={field.placeholder}
                                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                                      required={field.required}
                                    />
                                  )}
                                </div>
                              ))}
                            </div>
                          </div>
                        )}
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="text-center py-8">
                    <p className="text-gray-500">No payment methods available</p>
                  </div>
                )}
              </div>
            )}
          </div>

          {/* Footer */}
          <div className="p-4 md:p-6 border-t border-gray-100 bg-white flex-shrink-0 shadow-lg">
            {/* Mobile Layout */}
            <div className="md:hidden">
              <div className="text-right mb-4">
                <p className="text-sm text-gray-600">Total</p>
                <p className="text-2xl font-bold text-blue-600">{formatCurrency(total, storeSettings, currencies)}</p>
              </div>
              <div className="flex items-center justify-between gap-4">
                <div>
                  {step > 1 && (
                    <button
                      onClick={handlePrevStep}
                      className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors cursor-pointer"
                    >
                      Back
                    </button>
                  )}
                </div>
                <div>
                  {step < 3 ? (
                    <button
                      onClick={handleNextStep}
                      disabled={
                        (step === 1 && (!customerInfo.firstName || !customerInfo.lastName || !customerInfo.email || !customerInfo.phone || !customerInfo.address || !customerInfo.city || !customerInfo.state || !customerInfo.country || !customerInfo.postalCode)) ||
                        (step === 2 && !selectedShipping)
                      }
                      className="px-8 py-3 bg-blue-600 hover:bg-blue-700 disabled:bg-gray-300 text-white font-semibold rounded-lg transition-colors cursor-pointer disabled:cursor-not-allowed"
                    >
                      Continue
                    </button>
                  ) : (
                    <button
                      onClick={() => handlePlaceOrder(total)}
                      disabled={
                        isPlacingOrder ||
                        !selectedPayment ||
                        (selectedPayment === 'whatsapp' && !whatsappNumber.trim())
                      }
                      className="px-8 py-3 bg-blue-600 hover:bg-blue-700 disabled:bg-gray-300 text-white font-semibold rounded-lg transition-colors cursor-pointer disabled:cursor-not-allowed"
                    >
                      {isPlacingOrder ? 'Processing...' : 'Place Order'}
                    </button>
                  )}
                </div>
              </div>
            </div>
            
            {/* Desktop Layout */}
            <div className="hidden md:flex md:items-center md:justify-between gap-4">
              <div>
                {step > 1 && (
                  <button
                    onClick={handlePrevStep}
                    className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors cursor-pointer"
                  >
                    Back
                  </button>
                )}
              </div>
              <div className="flex items-center gap-4">
                <div className="text-right">
                  <p className="text-sm text-gray-600">Total</p>
                  <p className="text-xl font-bold text-blue-600">{formatCurrency(total, storeSettings, currencies)}</p>
                </div>
                {step < 3 ? (
                  <button
                    onClick={handleNextStep}
                    disabled={
                      (step === 1 && (!customerInfo.firstName || !customerInfo.lastName || !customerInfo.email || !customerInfo.phone || !customerInfo.address || !customerInfo.city || !customerInfo.state || !customerInfo.country || !customerInfo.postalCode)) ||
                      (step === 2 && !selectedShipping)
                    }
                    className="px-8 py-3 bg-blue-600 hover:bg-blue-700 disabled:bg-gray-300 text-white font-semibold rounded-lg transition-colors cursor-pointer disabled:cursor-not-allowed"
                  >
                    Continue
                  </button>
                ) : (
                  <button
                    onClick={() => handlePlaceOrder(total)}
                    disabled={
                      isPlacingOrder ||
                      !selectedPayment ||
                      (selectedPayment === 'whatsapp' && !whatsappNumber.trim())
                    }
                    className="px-8 py-3 bg-blue-600 hover:bg-blue-700 disabled:bg-gray-300 text-white font-semibold rounded-lg transition-colors cursor-pointer disabled:cursor-not-allowed"
                  >
                    {isPlacingOrder ? 'Processing...' : 'Place Order'}
                  </button>
                )}
              </div>
            </div>
          </div>
        </div>
      </div>

    </div>
  );
};

export const Checkout: React.FC<CheckoutProps> = ({ cartItems, currency, onClose, onOrderComplete, onUpdateCart, userProfile, isLoggedIn, onRemoveFromCart, onUpdateQuantity, onQuantityChange, store }) => {
  return (
    <CheckoutProvider userProfile={userProfile} isLoggedIn={isLoggedIn} store={store}>
      <CheckoutContent 
        cartItems={cartItems}
        currency={currency}
        onClose={onClose}
        onOrderComplete={onOrderComplete}
        onUpdateCart={onUpdateCart}
        onRemoveFromCart={onRemoveFromCart}
        onUpdateQuantity={onUpdateQuantity}
        onQuantityChange={onQuantityChange}
      />
    </CheckoutProvider>
  );
};